/* ============================================================================
 *
 * ADC_DelSig_Demo.c
 *
 * This file contains the code for the delta-sigma ADC demo task.
 * It also includes a local function to set up the LCD when the demo starts.
 *
 * v1.0	YFS	Initial release.
 *
 * ============================================================================
*/
 
/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <event_groups.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>
#include <Task_Defs.h>

/*
 * Function:	Print_ADC_DelSig_LCD_Template
 *
 * Prepare the LCD to show the demo results, as below.
 *
 * +----------------+
 * |ADC_DelSig Demo |
 * |DelSig =        |
 * +----------------+
 *
 * Globals:		None
 * 
 * Parameters:	None
 *
 * Return:		void
 */
 void Print_ADC_DelSig_LCD_Template()
{
	/* Top line of LCD */
	LCD_ClearDisplay();
	LCD_PrintString( "ADC DelSig demo" );
	
	/* Second line of LCD */
	LCD_Position( BOTTOM_ROW, ROW_START );
	LCD_PrintString( "DelSig =" );
} /* Print_ADC_DelSig_LCD_Template() */

/*
 * Function:		ADC_DelSig_Demo_Task
 *
 * This function runs as a FreeRTOS task with a medium priority. It is a
 * demonstration of how to use the Delta-Sigma ADC to read the voltage on 
 * a pin and print the result to the LCD.
 *
 * Note that this demo has a visible initialization sequence (scrolling
 * instructions on the LCD) and so an extra wait call on the task event bit
 * is used to suspend the task until the user requests the demo. Without this
 * call the user would see the instructions before asking to run the demo!
 *
 * Execution of this task is controlled by an event group. After creating
 * this task, the main task starts the demo by setting the bit in Demo_Events
 * that corresponds to this demo. When the user signals a desire to end the
 * demo, it stops the ADC and then waits on the event bit again. This leaves
 * the task ready to be re-executed at a later date.
 *
 *			Start task
 *			Wait on its event bit (wait for user to request demo)
 *			Print wiring instructions on LCD
 *			Wait for button press
 *			Clear LCD and print demo text
 *			Start the ADC
 *		 -> Read the ADC, convert the result to millivolts, print to LCD
 *		|   Check if SW3 is pressed. If so...
 *		|   	Stop the ADC
 *      |       Set the Main_Task event bit (make it ready to run)
 *      |   	Wait on its event bit (sleep)
 *      |   	Clear LCD and print demo text
 *		|		Restart the ADC
 *		 -- Loop
 * 
 * Globals:		Uses Demo_Events and Main_Task_Event
 *
 * Parameters:	pvParameters (demo index number)
 *
 * Return:		None (infinite loop)
 */
 void ADC_DelSig_Demo_Task( void *pvParameters )
{
	int demoNumber = (int)pvParameters;     /* Passed from xTaskCreate */
	
	int32 adc_val;	                        /* ADC count and millivolt values */
	
	/*
	This task requires no real initialization and so just waits on the event.
	This allows Main_Task to show the menus and start this demo by setting the
	event bit later.
	*/
    
    /* Wait on the demo event bit so that Main_Task can run */
    xEventGroupWaitBits(
        Demo_Events,            /* Wait for this demo to be requested */
        1 << demoNumber,        /* Event to wait - each demo owns a bit in the group */
        pdTRUE,                 /* Clear the event on exit */
        pdFALSE,                /* Do not wait for all bits (irrelevant) */
        WAIT_FOREVER );
	
	/*
	We woke up so Main_Task has set the event and it is time to run the demo.
	Prepare the LCD to show the demo results and start the ADC conversions.
	*/
	Print_ADC_DelSig_LCD_Template();
	
	CyDelay( 250 );	/* Debounce (250ms) press that started the demo */
	
	/* Display the breadboard wiring instructions */
	scroll( BOTTOM_ROW, "Attach \"VR\" to P0[0] on Dev Kit" );
	
	/* Wait for the user to make the connection on the breadboard */
	LCD_Position( BOTTOM_ROW, ROW_START );
	LCD_PrintString( "SW2 to continue" );
	while( P6_1_Read() )
	{
		/* Do nothing - wait for the button press */
	}
	
	/* Refresh the display (again) so the demo can finally start */
	Print_ADC_DelSig_LCD_Template();	
	
    /* Turn on the ADC */
    ADC_DelSig_Start();
    ADC_DelSig_StartConvert();
	
	for( ; ; )		/* Demo main loop (forever) */
	{
		/* Get a reading from the ADC and convert it to millivolts */
		ADC_DelSig_IsEndConversion( ADC_DelSig_WAIT_FOR_RESULT );
        adc_val = ADC_DelSig_GetResult32();
		adc_val = ADC_DelSig_CountsTo_mVolts( adc_val );
		
		/* Print the millivolts to the LCD - allow room for "DelSig = " */
		LCD_Position( BOTTOM_ROW, 9 );
		LCD_PrintNumber( adc_val );
		LCD_PrintString( "mV " );
		
		/* Check to see if SW3 was pressed - this signals the end of the demo */
		if( ! P15_5_Read() )		/* Note: 0 (ground) means SW3 was pressed */
		{
			/* Stop the ADC - good neighbor policy - another task may want it */
			ADC_DelSig_Stop();
			
			/* Set the MainTask event bit to make it ready to run */
			xEventGroupSetBits( Main_Task_Event, MAIN_TASK_EVENT_NUM );
			
            /* Wait on the demo event bit so that Main_Task can run */
            xEventGroupWaitBits(
                Demo_Events,            /* Wait for this demo to be requested */
                1 << demoNumber,        /* Event to wait - each demo owns a bit in the group */
                pdTRUE,                 /* Clear the event on exit */
                pdFALSE,                /* Do not wait for all bits (irrelevant) */
                WAIT_FOREVER );
			
			/*
			Main_Task set the event bits so it is time to run the demo again.
			Prepare the LCD and start the ADC conversions.
			*/
			Print_ADC_DelSig_LCD_Template();
	
			ADC_DelSig_Start();
			ADC_DelSig_StartConvert();

		} /* if( ! P15_5_Read() ) */

	} /* for( ; ; ) */

} /* ADC_DelSig_Demo_Task() */
